#ifndef AHLGREN_NODE
#define AHLGREN_NODE

#include <string>
#include <sstream> // istringstream
#include <cctype> // isupper, isdigit
#include <algorithm> // for_each
#include <utility> // pair (is_function)
#include <cassert>
#include <functional> // function<> (collect)
#include <cmath> // pow
#include <stack> // depth_iterator
#include <queue> // breadth_iterator
#include <map>
#include <set>
#include <new> // overloading new/delete for functor
#include <iterator>

//#include "tokenizer.h" // dictionary, Functor::create_infix
//#include "substitution.h" // Functor::subst
#include "useful.h"
#include "global.h"
#include "functor_tbl.h"

namespace lp {
	using namespace std;
	using namespace useful;

	void print_memory_stat(); // for debugging memory leaks

	class Mode;
	class Subst;
	class Dictionary;


	class Functor {
	public:
		// id_type: - for variables, + for other functors, 0 for nil
		typedef vector<Functor*> vec_type;
		typedef vec_type::size_type arity_type;
		typedef vec_type::iterator arg_iterator;
		typedef vec_type::const_iterator arg_const_iterator;
		typedef vec_type::reverse_iterator arg_reverse_iterator;
		typedef vec_type::const_reverse_iterator arg_const_reverse_iterator;
		// typedef unsigned long long var_index_type;
		class position; // position of a functor in expression
		// error class for evaluation
		class eval_error {
			string err;
		public:
			eval_error(const string& e) : err(e) {}
			string what() const { return err; }
		};

		//======================== Functor Iterators =======================//
		class depth_iterator; // preorder depth first search
		depth_iterator depth_begin();
		depth_iterator depth_end();
		class depth_const_iterator; // depth first search (const version)
		depth_const_iterator depth_begin() const;
		depth_const_iterator depth_end() const;
		depth_const_iterator depth_cbegin() const;
		depth_const_iterator depth_cend() const;
		// Postorder traversal
		class post_iterator;
		post_iterator post_begin();
		post_iterator post_end();
		class post_const_iterator;
		post_const_iterator post_begin() const;
		post_const_iterator post_end() const;
		post_const_iterator post_cbegin() const;
		post_const_iterator post_cend() const;
		// p/1 arg_first iterators
		class p1f_iterator; // preorder depth first search
		p1f_iterator p1f_begin();
		p1f_iterator p1f_end();
		class p1f_const_iterator; // depth first search (const version)
		p1f_const_iterator p1f_begin() const;
		p1f_const_iterator p1f_end() const;
		p1f_const_iterator p1f_cbegin() const;
		p1f_const_iterator p1f_cend() const;
		// Default to depth first search
		depth_iterator begin();
		depth_iterator end();
		depth_const_iterator begin() const;
		depth_const_iterator end() const;
		depth_const_iterator cbegin() const;
		depth_const_iterator cend() const;

		//======================== Sequence Iterators =======================//
		template <typename Fun, id_type Id> class Seq_iterator;
		// Instantiate template for sequences and lists
		typedef Seq_iterator<const Functor,sequence_id> seq_const_iterator;
		typedef Seq_iterator<Functor,sequence_id> seq_iterator;
		typedef Seq_iterator<const Functor,pair_id> list_const_iterator;
		typedef Seq_iterator<Functor,pair_id> list_iterator;
		seq_const_iterator seq_cbegin() const;
		seq_const_iterator seq_cend() const;
		seq_const_iterator seq_begin() const;
		seq_const_iterator seq_end() const;
		seq_iterator seq_begin();
		seq_iterator seq_end();
		list_const_iterator list_cbegin() const;
		list_const_iterator list_cend() const;
		list_const_iterator list_begin() const;
		list_const_iterator list_end() const;
		list_iterator list_begin();
		list_iterator list_end();

#ifndef NDEBUG
		// Overloading new and delete for detecting memory leaks
		void* operator new(size_t);
		void operator delete(void*);
#endif

		// Create nil Functor
		Functor() : _id(0), _par(nullptr) {}
		// Name constructors: create from numeric value
		static Functor make_int(long long val) { return Functor(fmap.id(val)); }
		static Functor make_double(double val) { return Functor(fmap.id(val)); }
		// Create Functor with id (will not be stored in functor_map)
		Functor(id_type i) : _id(i), _par(nullptr) {}
		Functor(id_type i, Functor* c) : _id(i), _par(nullptr) { steal(c); }
		Functor(id_type i, Functor* c, Functor* d) : _id(i), _par(nullptr) { steal(c); steal(d); }
		// Create Functor with symbol
		Functor(const string& symb) : _id(fmap.id(symb)), _par(nullptr) {}
		Functor(string&& symb) : _id(fmap.id(move(symb))), _par(nullptr) {}
		Functor(const char* symb) : _id(fmap.id(symb)), _par(nullptr) {}
		// Create Functor with symbol and a child
		Functor(const string& s, Functor* c) : _id(fmap.id(s)), _par(nullptr) { steal(c); }
		Functor(string&& s, Functor* c) : _id(fmap.id(move(s))), _par(nullptr) { steal(c); }
		// Create Functor with symbol and a child
		Functor(const string& s, Functor* c1, Functor* c2) : _id(fmap.id(s)), _par(nullptr) { steal(c1); steal(c2); }
		Functor(string&& s, Functor* c1, Functor* c2) : _id(fmap.id(move(s))), _par(nullptr) { steal(c1); steal(c2); }
		// Create functor from allocated functor pointer
		explicit Functor(Functor* f);
		// Create functor from data structure
		Functor(const data& d) : _id(fmap.id(d)), _par(nullptr) { }
		// Destructor
		~Functor();

		// Clear all children
		void clear();

		// Copy/Move constructors/assignments
		Functor(const Functor&);
		Functor(Functor&&);
		// Note: assignment operators do not alter the parent of 'this', use detach() if necessary
		Functor& operator=(const Functor&);
		Functor& operator=(Functor&&);

		// Is *this and *f syntactically equivalent?
		bool operator==(const Functor& f) const;
		bool operator!=(const Functor& f) const { return !(*this == f); }
		// Is *this a renaming of *f?
		bool is_renaming(const Functor* f) const { map<id_type,id_type> m,r; return is_renaming(f,m,r); }
		// Same as above, but store substitutions (rvs stores the inverse substitutions)
		bool is_renaming(const Functor* f, map<id_type,id_type>& vs, map<id_type,id_type>& rvs) const;
		// Lexicographical comparison of *this and f
		int compare(const Functor* f) const;
		bool operator<(const Functor& f) const { return this->compare(&f) < 0; }
		bool operator<=(const Functor& f) const { return this->compare(&f) <= 0; }
		bool operator>(const Functor& f) const { return this->compare(&f) > 0; }
		bool operator>=(const Functor& f) const { return this->compare(&f) >= 0; }
		// Lexicographical comparison of *this and f, expanding variables using substitutions
		int compare(const Functor* f, const Subst& subs) const;
		bool is_equal(const Functor* f, const Subst& s) const { return compare(f,s) == 0; }
		// Compare using Prolog standard ordering (less efficient)
		int compare_prolog(const Functor*) const;

		// Get/Set Functor symbol and number of children
		id_type id() const { return _id; }
		void id(id_type i) { _id = i; }
		arity_type arity() const { return argv.size(); }
		bool is_leaf() const { return argv.empty(); }
		// Does functor have signature i/a?
		bool sign(id_type i, arity_type a) const { return id() == i && arity() == a; }
		Functor* top(); // get top of tree
		const Functor* top() const { return const_cast<Functor*>(this)->top(); }
		// Get signature
		std::pair<lp::id_type,int> signature() const { return std::make_pair(id(),arity()); }

		//====================== Functor types =======================//
		bool is_nil() const { return id() == 0; }
		bool is_variable() const { assert(id() < 0 || argv.empty()); return id() > 0; }
		bool is_variable(id_type v) const { return argv.empty() && id() == v; }
		bool is_reserved_variable() const { return fmap.is_reserved_var(id()); } 
		// Check if node is a variable or bound to a variable
		bool is_variable(const Subst& s) const;
		// Check if node is a variable that is not bound (not even to another variable)
		bool is_free_variable(const Subst& s) const;
		bool is_constant() const { return argv.empty() && !is_variable(); }
		bool is_constant(id_type i) const { return argv.empty() && i == id(); }
		bool is_constant(const Subst& s) const;
		bool is_numeric() const;
		bool is_numeric(const Subst& s) const;
		bool is_int() const;
		bool is_int(const Subst& s) const;
		bool is_double() const;
		bool is_double(const Subst& s) const;
		bool is_function() const { return !argv.empty() && !is_variable(); }
		bool is_function(id_type i, int a) const { return id() == i && argv.size() == a; }
		bool is_function(const Subst& s) const;
		bool is_empty_list() const { return id() == empty_list_id && is_leaf(); }
		bool is_pair() const { return id() == pair_id && arity() == 2; }
		bool is_seq() const { return id() == sequence_id && arity() == 2; }
		bool is_rule() const { return id() == if_id && arity() == 2; }
		bool is_goal() const { return id() == if_id && arity() < 2; } // empty clause is goal
		bool is_nonempty_goal() const { return id() == if_id && arity() == 1; }
		bool is_empty_goal() const { return id() == if_id && is_leaf(); } // empty clause
		bool is_fact() const { return id() != if_id || arity() > 2; } // note: ':-'/3 is a fact too
		bool is_pos_ex() const { return is_ground() && is_fact(); }
		bool is_pos_ex(const id_type& i, arity_type a) const { return is_pos_ex() && id() == i && arity() == a; }
		bool is_neg_ex() const { return is_ground() && is_nonempty_goal() && !arg_last()->is_seq(); } // 1 body literal
		bool is_neg_ex(const id_type& i, arity_type a) const { return is_neg_ex() && arg_first()->id() == i && arg_first()->arity() == a; }
		//======================= Conversion methods ======================//
		// Convert functor to symbol (throws conversion_error)
		const char* to_string() const { return fmap.get_atom(id()); }
		const char* to_string(const Subst&) const;
		// Convert functor to unsigned/integer/double value (throws non_numeric)
		long long to_int() const { return fmap.get_int(id()); }
		long long to_int(const Subst&) const;
		long long to_unsigned() const;
		long long to_unsigned(const Subst&) const;
		double to_double() const { return fmap.get_float(id()); }
		double to_double(const Subst& subs) const;
		// Convert to proper numeric type store in data
		data to_numeric() const { return fmap.get_numeric(id()); }
		data to_numeric(const Subst&) const;
		// Evaluate arithmetic expression
		data eval() const;
		// Evaluate arithmetic expression using substitutions
		data eval(const Subst&) const;

		//======================= Parent/Children methods ======================//
		// Get parent
		const Functor* parent() const { return _par; }
		Functor*& parent() { return _par; }
		// Get first argument
		const Functor* arg_first() const { return argv.front(); }
		Functor* arg_first() { return argv.front(); }
		// Get last argument
		const Functor* arg_last() const { return argv.back(); }
		Functor* arg_last() { return argv.back(); }
		// Pop first/last argument
		void arg_pop_first() { this->arg_erase(0); }
		void arg_pop_last() { delete argv.back(); argv.pop_back(); }
		// Erase argument at index/iterator k
		void arg_erase(arity_type k) { delete argv[k]; argv.erase(argv.begin()+k); }
		arg_iterator arg_erase(arg_iterator i) { delete *i; return argv.erase(i); }
		// Get argument at position i
		const Functor* arg(arity_type i) const { return argv[i]; }
		Functor* arg(arity_type i) { return argv[i]; }
		// Get argument iterators
		arg_iterator arg_begin() { return argv.begin(); }
		arg_iterator arg_end() { return argv.end(); }
		arg_const_iterator arg_begin() const { return argv.begin(); }
		arg_const_iterator arg_end() const { return argv.end(); }
		arg_reverse_iterator arg_rbegin() { return argv.rbegin(); }
		arg_reverse_iterator arg_rend() { return argv.rend(); }
		arg_const_reverse_iterator arg_rbegin() const { return argv.rbegin(); }
		arg_const_reverse_iterator arg_rend() const { return argv.rend(); }
		// Steal subtree and put it at i, return new arity of this
		void steal(Functor* s, arg_iterator i);
		void steal(Functor* s); // provides optimized version for inserting in back
		void steal(Functor* s, unsigned i) { return steal(s,argv.begin()+i); }
		void steal_front(Functor* s) { return steal(s,arg_begin()); }
		// Clear all arguments
		void arg_clear() { for_each(arg_begin(),arg_end(),[](Functor* c){delete c;}); argv.clear(); }

		//==================== Seaches and Replacements ====================//
		// Get functor node at position
		Functor* get(const position& p) { return const_cast<Functor*>(const_cast<const Functor*>(this)->get(p)); }
		const Functor* get(const position& p) const;
		// Count occurences of term satisfying Pred
		template <typename Pred> int count(Pred p) const;
		// Count number of functors in tree
		int size() const;
		// Replace this node by trg
		void replace_this(const Functor* trg);
		// Perform variable substitution (replace() for variables)
		void subst(const Subst&);
		void subst(id_type v, const Functor* f);
		void subst(id_type v, id_type w); // substitute variables: v -> w

		//================= Recursive Methods =================//
		// Get copy of whole subtree
		Functor* copy() const;
		// Is expression ground?
		bool is_ground() const;
		// Rename all variables to _Gx
		void rename() { map<id_type,id_type> m; return rename(m); }
		// Rename all variables as above and keep subs
		void rename(map<id_type,id_type>& subs);
		// Rename as above and keep both subs and inverted subs
		void rename(map<id_type,id_type>& subs, map<id_type,id_type>& inv_subs);
		// Reverse all renamings
		void unrename(const map<id_type,id_type>& inv_subs);

		// Pretty-printing methods
		// Give _Gx variables nice, standard names (this is important to avoid variable clashes too)
		// void unrename();

		// Get all constants/functions
		template <typename Pred> void collect(set<id_type>& c, Pred pred) const;
		template <typename Pred> void collect(set<pair<id_type,arity_type> >& c, Pred pred) const;
		void constants(set<id_type>& c) const { collect(c,[](const Functor* f){return f->is_constant();}); }
		void variables(set<id_type>& c) const { collect(c,[](const Functor* f){return f->is_variable();}); }
		// Collect functions, excluding constants
		void functions(set<pair<id_type,arity_type> >& c) const { collect(c,[](const Functor* f){return f->is_function();}); }
		// Collect functions and Constants
		void functors(set<pair<id_type,arity_type> >& c) const { collect(c,[](const Functor* f){return !f->is_variable();}); }
		// Flatten/unflatten expression
		//void flatten(vector<Functor*>& fp);
		//void flatten(Functor* lit, vector<Functor*>& flat_preds);
		//void unflatten(const vector<Functor*>& fp);

		// I/O operations and Dictionary related methods
		// Convert any id_type to data
		static data get_data(id_type i) { return fmap.get_data(i); }
		// Convert any name to an ID
		static id_type index(const char* s) { return fmap.id(s); }
		static id_type index(const string& s) { return fmap.id(s); }
		static id_type index(string&& s) { return fmap.id(move(s)); }
		static id_type index(long long i) { return fmap.id(i); }
		static id_type index(double d) { return fmap.id(d); }

		// Read from stream in infix notation, using dictionary
		static Functor* read_infix(istream& is, long long& linec, const Dictionary& dic = prolog_dic);
		static Functor* read_infix(istream& is, const Dictionary& dic = prolog_dic) { long long l = 1; return read_infix(is,l,dic); }

		// Print id as proper symbol (Helper for external functions)
		static void print_id(ostream& os, id_type i) { fmap.print(os,i); }

		// Helper to print only this symbol: ostream, current printable var name, id->var mapping
		void print_this(ostream&, string&, map<id_type,string>&) const; // use static pretty_print_level
		static void next_letter(string& s, const Functor& expr); // get next letter s that is not in expr

		void print_infix(ostream& os, const Dictionary& dic) const;
		void print_infix(ostream& os, string& vn, map<id_type,string>& vm, const Dictionary& dic = prolog_dic) const;
		void print_prefix(ostream&) const;
		int precedence(const Dictionary& dic) const;
		bool is_prefix_operator(const Dictionary& dic) const;
		bool is_infix_operator(const Dictionary& dic) const;
		bool is_suffix_operator(const Dictionary& dic) const;


		//======================== Clause Methods =======================//
		// Number of literals in clause
		arity_type literals() const;

		// Head manipulation
		void head_clear();
		const Functor* head() const; // get front of head
		Functor* head() { return const_cast<Functor*>(const_cast<const Functor*>(this)->head()); }
		// Create new head
		void head(Functor* h);
		// Get body (sequence)
		const Functor* body() const;
		Functor* body() { return const_cast<Functor*>(const_cast<const Functor*>(this)->body()); }
		// Create new body
		void body(Functor* b);
		// Get Body literal k
		//const Functor* body(arity_type k) const;
		//Functor* body(arity_type k) { return const_cast<Functor*>(const_cast<const Functor*>(this)->body(k)); }
		// Get number of literals in body
		arity_type body_size() const;
		bool body_empty() const { return id() != if_id || (arity() != 1 && arity() != 2); }
		// Get iterator to body elements
		seq_const_iterator body_begin() const;
		seq_const_iterator body_end() const;
		seq_iterator body_begin();
		seq_iterator body_end();
		// Insertion/Deletion
		void body_clear();
		const Functor* body_front() const;
		Functor* body_front();
		const Functor* body_back() const;
		Functor* body_back();
		void body_pop_front();
		void body_pop_back();
		// Erase literal i
		seq_iterator body_erase(seq_iterator i);
		// Release front/back
		Functor* body_release_front();
		Functor* body_release_back();
		// Release pointer to i
		Functor* body_release(seq_iterator i);
		// Release entire body sequence
		Functor* body_release();

		//void body_erase(arity_type k) { body_erase(body_begin()+k); }
		void body_push_front(Functor* f);
		void body_push_back(Functor* f, Functor* last = nullptr);
		void body_insert(Functor* f, seq_iterator pos);

		// Is this a tautology?
		bool is_tautology() const;

		// Is clause head connected? (crude form of I/O compliance checking)
		bool is_ioconnected() const;

		// Flatten body by applying equalities (=/2 and _=/2)
		void flatten();

		//======================== Unique Variables Helpers =======================//
		// Get unique variable name
		static id_type unique_index() { return fmap.new_var_id(); }

		//======================== Low Level Methods =======================//
		// Check that parent points correctly point back
		bool check_parents() const;
		// Reserve space for children
		void arg_reserve(arity_type k) { argv.reserve(k); }
		// Steal all children from p
		void steal_all(Functor* p);
		// Remove args but don't delete them
		void arg_release() { argv.clear(); }
		// Detach this node from it's parent, set this nodes parent to nullptr, return this
		Functor* detach();
		// Attach a child to this functor that will NOT point back
		arity_type attach_child(Functor* f) { argv.push_back(f); return argv.size()-1; }
		// Unattach a previously attached node
		void unattach_child() { argv.pop_back(); }
		void unattach_child(arity_type i) { argv.erase(argv.begin()+i); }
	protected:
		id_type _id; // id
		Functor* _par; // parent pointer
		vec_type argv; // children pointers
		// Remove n's parent's pointer to n
		static void remove_parent_pointer(Functor* n);
	};

	//******** Global Helpers ********//

	inline ostream& operator<<(ostream& os, const Functor& f)
	{
		f.print_infix(os,prolog_dic);
		return os;
	}
	
	// Function Object for equality testing
	 struct functor_eq {
		bool operator()(const Functor& f, const Functor& g) const { return f == g; }
	};
	 struct functor_ptr_eq {
		bool operator()(const Functor* f, const Functor* g) const { return *f == *g; }
	};

	// Function Object for rename-equality testing
	 struct functor_req {
		 bool operator()(const Functor& f, const Functor& g) const { return f.is_renaming(&g); }
	 };
	 struct functor_ptr_req {
		 bool operator()(const Functor* f, const Functor* g) const { return f->is_renaming(g); }
	 };
	// Hash function for functor expressions, indifferent about variables names
	unsigned long long hash_variant(const Functor& t, std::map<id_type,int>& vmap);
	// Function Object for hashing variants
	struct functor_hash_variant {
		std::size_t operator()(const Functor& f) const { 
			std::map<id_type,int> vm;
			return std::size_t(hash_variant(f,vm));
		}
	};

	 // Function Object for fast total ordering
	 /*
	 Two function symbols f/n and g/m are compared based on lexicographic (n,f) < (m,g)
	 If they're equal, recursively compare depth first.
	 */
	 struct functor_less {
		 bool operator()(const Functor& f, const Functor& g) const { return f.compare(&g) < 0; }
	 };
	 struct functor_ptr_less {
		 bool operator()(const Functor* f, const Functor* g) const { return f->compare(g) < 0; }
	 };

	 struct functor_ptr_size {
		 bool operator()(const Functor* f, const Functor* g) const {
			 const auto sf = f->size();
			 const auto sg = g->size();
			 if (sf < sg) return true;
			 if (sf > sg) return false;
			 return functor_ptr_less()(f,g);
		 }
	 };

	 // Sort by functor complexity
	 struct functor_size {
		 bool operator()(const Functor& f, const Functor& g) const {
			 return functor_ptr_size()(&f,&g);
		 }
	 };

	// Sort functors according to prolog ordering (less efficient than standard compare)
	struct prolog_order_ptr {
		bool operator()(const Functor* f, const Functor* g) const {
			return f->compare_prolog(g) < 0;
		}
	};

	 struct prolog_order {
		 bool operator()(const Functor& f, const Functor& g) const {
			 return prolog_order_ptr()(&f,&g);
		 }
	 };

	//******** Global Functions ********//

	// Output functor expression
	ostream& operator<<(ostream& os, const Functor& f);
	
	// Make functor from string
	Functor* make_functor_ptr(const string& s, const Dictionary& dic = prolog_dic);	
	Functor make_functor(const string& s, const Dictionary& dic = prolog_dic);

	// Make most general functor with arity
	Functor* make_general(id_type s, Functor::arity_type a);

	// Check if a functor is an ordered subset of another
	//bool is_subset(const Functor& f, const Functor& g, bool ordered = false);
	//bool is_renamed_subset(const Functor& f, const Functor& g, bool ordered = false);



	//******** Templated members ********//

	template <typename Pred>
	void Functor::collect(set<id_type>& c, Pred pred) const
	{
		if (pred(this)) c.insert(id());
		for_each(arg_begin(),arg_end(),[&](const Functor* f){f->collect(c,pred);});
	}

	
	template <typename Pred>
	void Functor::collect(set<pair<id_type,arity_type> >& c, Pred pred) const
	{
		if (pred(this)) c.insert(make_pair(id(),arity()));
		for_each(arg_begin(),arg_end(),[&](const Functor* f){f->collect(c,pred);});
	}


	//template <typename Pred>
	//const Functor* Functor::find(Pred p) const
	//{
	//	std::find_if(

	//	if (p(this)) return this;
	//	const Functor* f;
	//	for (auto i = arg_begin(); i != arg_end(); ++i) {
	//		 if ((f = (*i)->find(p)) != nullptr) return f; // assign to f and return if not nullptr
	//	}
	//	return nullptr;
	//}


	template <typename Pred>
	int Functor::count(Pred p) const
	{
		int c = 0;
		std::for_each(begin(),end(),[&](const Functor& n){ c += p(n);  });
		return c;
	}


	//template <class Fun>
	//void Functor::apply(Fun f)
	//{
	//	std::for_each(post_begin(),post_end(),[&](Functor& n){ f(&n); });
	//}

	//template <class Fun>
	//void Functor::apply(Fun f) const
	//{
	//	std::for_each(post_begin(),post_end(),[&](const Functor& n){ f(&n); });
	//}


	//template <typename Rep>
	//int Functor::replace_if(Rep rep)
	//{
	//	Functor* r = rep(this);
	//	if (r) {
	//		// Switch symbol instead of removing node (to keep pointer address)
	//		id( r->id() );
	//		arg_clear(); // clear children
	//		// add new children
	//		steal_all(r);
	//		delete r;
	//		return 1;
	//	} else {
	//		int n = 0;
	//		for_each(arg_begin(),arg_end(),[&](Functor* c){ n += c->replace_if(rep); });
	//		return n;
	//	}
	//}





	// ======================== Position of functors ======================== //
	
	class Functor::position {
	public:
		typedef vector<arity_type>::size_type index_type;
		typedef vector<arity_type>::iterator iterator;
		typedef vector<arity_type>::const_iterator const_iterator;
		// Top position
		position() {}
		position(const position& p) : pos(p.pos) {}
		position& operator=(const position& p) { if (this != &p) pos = p.pos; return *this; }
		// Move constructor/assignment
		position(position&& p) : pos(move(p.pos)) { }
		position& operator=(position&& p) { if (this != &p) pos = std::move(p.pos); return *this; }

		// Create position for node f in its expression
		position(const Functor* f);
		// Create position from numerical values
		position(int a) : pos(1,static_cast<arity_type>(a)) {} // avoid confusion due to overloads
		position(arity_type a) : pos(1,a) {}
		position(arity_type a, arity_type b) : pos(1,a) { pos.push_back(b); }
		position(arity_type a, arity_type b, arity_type c) : pos(1,a) { pos.push_back(b); pos.push_back(c); }

		// Add to position
		position& operator+=(arity_type k) { pos.push_back(k); return *this; }
		position operator+(arity_type k) const { position p = *this; p.pos.push_back(k); return p; }
		// Remove from position
		void pop_back() { pos.pop_back(); }
		// Get branches of position
		arity_type operator[](index_type k) const { return pos[k]; }
		arity_type& operator[](index_type k) { return pos[k]; }
		// Iterators
		iterator begin() { return pos.begin(); }
		iterator end() { return pos.end(); }
		const_iterator begin() const { return pos.begin(); }
		const_iterator end() const { return pos.end(); }
		// Compare positions
		bool operator<(const position& p) const { return pos < p.pos; }
		bool operator==(const position& p) const { return pos == p.pos; }

		// Print
		friend std::ostream& operator<<(std::ostream& os, const position& p)
		{
			os << "(";
			auto i = p.begin();
			if (i != p.end()) {
				os << *i;
				for (++i; i != p.end(); ++i) os << "," << *i;
			}
			os << ")";
			return os;
		}
		// Get vector
		vector<arity_type>& get_vector() { return pos; }
	protected:
		vector<arity_type> pos;
	};



	//=========================== Iterator Classes ===========================//

	template <typename Fun, id_type Id>
	class Functor::Seq_iterator : public std::iterator<std::forward_iterator_tag,Fun> {
	public:
		Seq_iterator() : p(nullptr) {}
		Seq_iterator(nullptr_t) : p(nullptr) {}
		Seq_iterator(Fun* f) : p(nullptr) {
			if (f) {
				if (f->sign(Id,2)) p = f->arg_first();
				else p = f;
			} // else p = nullptr already set
		}
		bool operator==(const Seq_iterator& i) const { return p == i.p; } // compare pointers
		bool operator!=(const Seq_iterator& i) const { return p != i.p; } // compare pointers
		Seq_iterator& operator++() {
			assert(p != nullptr); // cannot move forward from end()
			Fun* par = p->parent();
			if (par && par->sign(Id,2) && par->arg_last() != p) {
				p = par->arg_last();
				if (p->sign(Id,2)) p = p->arg_first();
			} else p = nullptr; // singleton or last element, now at end
			return *this;
		}
		Seq_iterator operator++(int) { Seq_iterator t(*this); ++(*this); return t; }
		// Not a bidirectional iterator as we can't go back from end(), but otherwise fine
		Seq_iterator& operator--() {
			assert(p != nullptr); // cannot move back from end()
			Fun* par = p->parent();
			if (par && par->sign(Id,2)) {
				if (par->arg_first() == p) { 
					// not last element
					if (par->parent() && par->parent()->sign(Id,2)) {
						// not first element
						p = par->parent()->arg_first();
					} else {
						// first element
						p = nullptr;
					}
				} else {
					// last element
					assert(par->arg_last() == p);
					p = par->arg_first();
				}
			} else {
				// p is singleton (p has no parent, or parent is not ,/2)
				p = nullptr;
			}
			return *this;
		}
		Seq_iterator operator--(int) { Seq_iterator t(*this); --(*this); return t; }
		Fun& operator*() const { return *p; }
		Fun* operator->() const { return p; }
		// Convert seq_iterator to seq_const_iterator
		operator Seq_iterator<const Functor,Id>() const { return Seq_iterator<const Functor,Id>(p); }
	protected:
		Fun* p;
	};



	//class Functor::breadth_const_iterator : public std::iterator<std::forward_iterator_tag, Functor> {
	//public:
	//	typedef forward_iterator_tag iterator_category;
	//	typedef Functor value_type;
	//	typedef value_type* pointer;
	//	typedef value_type& reference;

	//	// Constructor: point at end
	//	breadth_const_iterator();
	//	breadth_const_iterator(nullptr_t);
	//	// Construct iterator pointing to functor
	//	breadth_const_iterator(const Functor* ftor);
	//	// Prefix increment
	//	breadth_const_iterator& operator++();
	//	// Postfix increment
	//	breadth_const_iterator operator++(int);
	//	// Comparison
	//	bool operator==(const breadth_const_iterator& di) const { return nptr == di.nptr; }
	//	bool operator!=(const breadth_const_iterator& di) const { return !(*this == di); }
	//	const Functor& operator*() const { return *nptr; }
	//	const Functor* operator->() const { return nptr; }
	//protected:
	//	const Functor* nptr;
	//	queue<const Functor*> q;
	//};
	//
	//class Functor::breadth_iterator : public Functor::breadth_const_iterator {
	//public:
	//	breadth_iterator() : breadth_const_iterator() { }
	//	breadth_iterator(nullptr_t) : breadth_const_iterator(nullptr) { }
	//	breadth_iterator(Functor* ftor) : breadth_const_iterator(ftor) { }
	//	bool operator==(const breadth_iterator& di) const { return this->nptr == di.nptr; }
	//	bool operator!=(const breadth_iterator& di) const { return !(*this == di); }
	//	Functor& operator*() { return const_cast<Functor&>(*this->nptr); }
	//	Functor* operator->() { return const_cast<Functor*>(this->nptr); }
	//};


// #define AHLGREN_FUNCTOR_FAT_ITERATORS	

#if defined AHLGREN_FUNCTOR_FAT_ITERATORS	
	class Functor::depth_const_iterator : public std::iterator<std::forward_iterator_tag, Functor> {
	public:
		typedef forward_iterator_tag iterator_category;
		typedef Functor value_type;
		typedef value_type* pointer;
		typedef value_type& reference;

		depth_const_iterator() : nptr(nullptr) { }
		depth_const_iterator(nullptr_t) : nptr(nullptr) { }

		depth_const_iterator(const Functor* ftor) : nptr(ftor) {
			if (!nptr) return;
			for_each(nptr->arg_rbegin(),nptr->arg_rend(),[&](const Functor* f){ stk.push(f); });
		}

		// Prefix increment
		depth_const_iterator& operator++() {
			if (stk.empty()) {
				// point at end()
				nptr = nullptr;
				return *this;
			}
			nptr = stk.top();
			stk.pop();
			for_each(nptr->arg_rbegin(),nptr->arg_rend(),[&](const Functor* f){ stk.push(f); });
			return *this;
		}

		// Postfix increment
		depth_const_iterator operator++(int) {
			depth_const_iterator tmp = *this;
			++(*this);
			return tmp;
		}

		bool operator==(const depth_const_iterator& di) const { return nptr == di.nptr; }
		bool operator!=(const depth_const_iterator& di) const { return !(*this == di); }
		const Functor& operator*() const { return *nptr;  }
		const Functor* operator->() const { return nptr; }
	protected:
		const Functor* nptr;
		stack<const Functor*> stk;
	};


	
	class Functor::depth_iterator : public Functor::depth_const_iterator {
	public:
		typedef forward_iterator_tag iterator_category;
		typedef Functor value_type;
		typedef value_type* pointer;
		typedef value_type& reference;
		depth_iterator() : depth_const_iterator() { }
		depth_iterator(nullptr_t) : depth_const_iterator(nullptr) { }
		depth_iterator(Functor* Functor) : depth_const_iterator(Functor) { }
		bool operator==(const depth_iterator& di) const { return this->nptr == di.nptr; }
		bool operator!=(const depth_iterator& di) const { return !(*this == di); }
		reference operator*() { return const_cast<Functor&>(*this->nptr); }
		pointer operator->() { return const_cast<Functor*>(this->nptr); }
	};
#else
	class Functor::depth_const_iterator : public std::iterator<std::bidirectional_iterator_tag,Functor> {
	public:
		typedef std::bidirectional_iterator_tag iterator_category;
		typedef Functor value_type;
		typedef value_type* pointer;
		typedef value_type& reference;

		// For begin: fp == rp == node
		depth_const_iterator(const Functor* fp) : nptr(fp), root(fp) { }
		// for end: fp == nullptr, rp == root
		depth_const_iterator(nullptr_t, const Functor* rp) : nptr(nullptr), root(rp) { }
		depth_const_iterator(const Functor* fp, const Functor* rp) : nptr(fp), root(rp) { }

		// Prefix increment
		depth_const_iterator& operator++() {
			//DEBUG_TRACE(cerr << "depth iter: points to " << *nptr << "\n");
			if (nptr->argv.empty()) {
				//DEBUG_TRACE(cerr << "depth iter: this is a leaf\n");
				// This is a leaf node, so we need to climb up
				for (;;) {
					if (nptr == root) {
						//DEBUG_TRACE(cerr << "depth iter: reached rootpar, setting to null\n");
						nptr = nullptr;
						break;
					}
					// note: if nptr is not root, it must have a parent
					const Functor* par = nptr->parent();
					//DEBUG_TRACE(cerr << "depth iter: parent is " << *par << "\n");
					// Determine which child we are
					// Note: tested loop unrolling and std::find, this is fastest:
					auto next = par->arg_begin();
					for ( ; *next != nptr ; ++next); // no bounds check: nptr is in array
					++next; // bring to next
					if (next != par->arg_end()) {
						// Branch down to next child
						nptr = *next;
						break;
					} else {
						// We were the last child of parent node, so continue up
						nptr = par;
					}
				}
			} else {
				// Not a leaf node, so move down one step to the left
				//DEBUG_TRACE(cerr << "depth iter: not a leaf\n");
				nptr = nptr->argv.front();
			}
			return *this;
		}

		// Postfix increment
		depth_const_iterator operator++(int) {
			depth_const_iterator tmp(*this);
			++(*this);
			return tmp;
		}

		// Prefix decrement
		depth_const_iterator operator--() {
			if (nptr) {
				// note: -- on first element is undefined => we may safely move up if not left
				assert(nptr->parent() != nullptr);
				if (nptr == nptr->parent()->arg_first()) {
					// nptr is first child => move up
					nptr = nptr->parent();
				} else {
					// nptr is not first child => move up one step, then traverse down
					// find pointer from parent
					auto prev = --nptr->parent()->arg_end();
					for ( ; *prev != nptr; --prev);
					--prev; // previous from nptr (prev can't be argv.front())
					nptr = *prev;
					// Now traverse down right most
					while (!nptr->is_leaf()) nptr = nptr->arg_last();
				}
			} else {
				// nptr at end, so we need to use root to get to last element
				for (nptr = root; !nptr->is_leaf(); ) {
					nptr = nptr->arg_last();
				}
			}
			return *this;
		}
		// Postfix decrement
		depth_const_iterator operator--(int) {
			depth_const_iterator tmp(*this);
			--(*this);
			return tmp;
		}

		bool operator==(const depth_const_iterator& di) const { return nptr == di.nptr; }
		bool operator!=(const depth_const_iterator& di) const { return !(*this == di); }
		const Functor& operator*() const { return *nptr;  }
		const Functor* operator->() const { return nptr; }
		const Functor* get_root() const { return root; }
	protected:
		const Functor* nptr;
		const Functor* root;
	private:
		depth_const_iterator(nullptr_t);
	};


	
	class Functor::depth_iterator : public Functor::depth_const_iterator {
	public:
		typedef std::bidirectional_iterator_tag iterator_category;
		typedef Functor value_type;
		typedef value_type* pointer;
		typedef value_type& reference;
		// For begin: fp == rp == node
		depth_iterator(const Functor* fp) : depth_const_iterator(fp) { }
		// for end: fp == nullptr, rp == root
		depth_iterator(nullptr_t, const Functor* rp) : depth_const_iterator(nullptr,rp) { }
		depth_iterator(const Functor* fp, const Functor* rp) : depth_const_iterator(fp,rp) { }
		bool operator==(const depth_iterator& di) const { return this->nptr == di.nptr; }
		bool operator!=(const depth_iterator& di) const { return !(*this == di); }
		reference operator*() { return const_cast<Functor&>(*this->nptr); }
		pointer operator->() { return const_cast<Functor*>(this->nptr); }
		Functor* get_root() { return const_cast<Functor*>(root); }
	private:
		depth_iterator(nullptr_t);
	};


	class Functor::post_const_iterator : public std::iterator<std::bidirectional_iterator_tag,Functor> {
	public:
		typedef std::bidirectional_iterator_tag iterator_category;
		typedef Functor value_type;
		typedef value_type* pointer;
		typedef value_type& reference;

		// For begin: fp == rp == node
		post_const_iterator(const Functor* fp) : nptr(fp), root(fp) { while (!nptr->is_leaf()) nptr = nptr->arg_first(); }
		// for end: fp == nullptr, rp == root
		post_const_iterator(nullptr_t, const Functor* rp) : nptr(nullptr), root(rp) { }
		post_const_iterator(const Functor* fp, const Functor* rp) : nptr(fp), root(rp) { }

		// Prefix increment
		post_const_iterator& operator++() {
			if (nptr == root) {
				nptr = nullptr;
			} else {
				// note: if nptr is not root, it must have a parent
				const Functor* par = nptr->parent();
				// Determine which child we are
				// Note: tested loop unrolling and std::find, this is fastest:
				auto next = par->arg_begin();
				for ( ; *next != nptr ; ++next); // no bounds check: nptr is in array
				++next; // bring to next
				if (next != par->arg_end()) {
					// Branch down to next child
					nptr = *next;
					while (!nptr->is_leaf()) nptr = nptr->arg_first();
				} else {
					// We were the last child of parent node, so next is parent
					nptr = par;
				}
			}
			return *this;
		}

		// Postfix increment
		post_const_iterator operator++(int) {
			post_const_iterator tmp(*this);
			++(*this);
			return tmp;
		}

		// Prefix decrement
		post_const_iterator operator--() {
			if (nptr) {
				if (nptr->is_leaf()) {
					// This is a leaf node, so we need to climb up
					for (;;) {
						if (nptr == root) {
							nptr = nullptr;
							break;
						}
						// note: if nptr is not root, it must have a parent
						const Functor* par = nptr->parent();
						// Determine which child we are
						// Note: tested loop unrolling and std::find, this is fastest:
						auto next = --par->arg_end();
						for ( ; *next != nptr ; --next); // no bounds check: nptr is in array
						if (next == par->arg_begin()) {
							// We were the last child of parent node, so continue up
							nptr = par;
						} else {
							// Branch down to next child
							--next; // bring to next
							nptr = *next;
							break;
						}
					}
				} else {
					// Not a leaf node, so move down one step to the right
					nptr = nptr->arg_last();
				}
			} else {
				nptr = root; // last element is root
			}
			return *this;
		}
		// Postfix decrement
		post_const_iterator operator--(int) {
			post_const_iterator tmp(*this);
			--(*this);
			return tmp;
		}

		bool operator==(const post_const_iterator& di) const { return nptr == di.nptr; }
		bool operator!=(const post_const_iterator& di) const { return !(*this == di); }
		const Functor& operator*() const { return *nptr;  }
		const Functor* operator->() const { return nptr; }
		const Functor* get_root() const { return root; }
	protected:
		const Functor* nptr;
		const Functor* root;
	private:
		post_const_iterator(nullptr_t);
	};


	
	class Functor::post_iterator : public Functor::post_const_iterator {
	public:
		typedef std::bidirectional_iterator_tag iterator_category;
		typedef Functor value_type;
		typedef value_type* pointer;
		typedef value_type& reference;
		// For begin: fp == rp == node
		post_iterator(const Functor* fp) : post_const_iterator(fp) { }
		// for end: fp == nullptr, rp == root
		post_iterator(nullptr_t, const Functor* rp) : post_const_iterator(nullptr,rp) { }
		post_iterator(const Functor* fp, const Functor* rp) : post_const_iterator(fp,rp) { }
		bool operator==(const post_iterator& di) const { return this->nptr == di.nptr; }
		bool operator!=(const post_iterator& di) const { return !(*this == di); }
		reference operator*() { return const_cast<Functor&>(*this->nptr); }
		pointer operator->() { return const_cast<Functor*>(this->nptr); }
		Functor* get_root() { return const_cast<Functor*>(root); }
	private:
		post_iterator(nullptr_t);
	};


	class Functor::p1f_const_iterator : public std::iterator<std::bidirectional_iterator_tag,Functor> {
	public:
		typedef std::bidirectional_iterator_tag iterator_category;
		typedef Functor value_type;
		typedef value_type* pointer;
		typedef value_type& reference;

		// For begin: fp == rp == node
		p1f_const_iterator(const Functor* fp) : nptr(fp), root(fp) { }
		// for end: fp == nullptr, rp == root
		p1f_const_iterator(nullptr_t, const Functor* rp) : nptr(nullptr), root(rp) { }
		p1f_const_iterator(const Functor* fp, const Functor* rp) : nptr(fp), root(rp) { }

		// Prefix increment
		p1f_const_iterator& operator++() {
			if (nptr->is_leaf()) {
				nptr = nullptr;
			} else {
				nptr = nptr->arg_first();
			}
			return *this;
		}

		// Postfix increment
		p1f_const_iterator operator++(int) {
			p1f_const_iterator tmp(*this);
			++(*this);
			return tmp;
		}

		// Prefix decrement
		p1f_const_iterator operator--() {
			if (nptr) {
				nptr = nptr->parent();
			} else {
				nptr = root;
				while (!nptr->is_leaf()) nptr = nptr->arg_first();
			}
			return *this;
		}
		// Postfix decrement
		p1f_const_iterator operator--(int) {
			p1f_const_iterator tmp(*this);
			--(*this);
			return tmp;
		}

		bool operator==(const p1f_const_iterator& di) const { return nptr == di.nptr; }
		bool operator!=(const p1f_const_iterator& di) const { return !(*this == di); }
		const Functor& operator*() const { return *nptr;  }
		const Functor* operator->() const { return nptr; }
		const Functor* get_root() const { return root; }
	protected:
		const Functor* nptr;
		const Functor* root;
	private:
		p1f_const_iterator(nullptr_t);
	};
	
	class Functor::p1f_iterator : public Functor::p1f_const_iterator {
	public:
		typedef std::bidirectional_iterator_tag iterator_category;
		typedef Functor value_type;
		typedef value_type* pointer;
		typedef value_type& reference;
		// For begin: fp == rp == node
		p1f_iterator(const Functor* fp) : p1f_const_iterator(fp) { }
		// for end: fp == nullptr, rp == root
		p1f_iterator(nullptr_t, const Functor* rp) : p1f_const_iterator(nullptr,rp) { }
		p1f_iterator(const Functor* fp, const Functor* rp) : p1f_const_iterator(fp,rp) { }
		bool operator==(const p1f_iterator& di) const { return this->nptr == di.nptr; }
		bool operator!=(const p1f_iterator& di) const { return !(*this == di); }
		reference operator*() { return const_cast<Functor&>(*this->nptr); }
		pointer operator->() { return const_cast<Functor*>(this->nptr); }
		Functor* get_root() { return const_cast<Functor*>(root); }
	private:
		p1f_iterator(nullptr_t);
	};

#endif



	//========================= Inline Definitions =========================//
	
} // namespace lp
#include "tokenizer.h"
#include "substitution.h"
namespace lp {

	inline Functor* Functor::read_infix(istream& is, long long& l, const Dictionary& dic) { return stream_to_expr(is,l,dic); }
	inline data Functor::eval() const { Subst s; return eval(s); }
	inline bool Functor::is_free_variable(const Subst& s) const { return is_variable() && !s.get(id()); }

	inline Functor::Functor(Functor* f) : _id(f->id()), _par(nullptr), argv(move(f->argv)) 
	{
		for_each(argv.begin(),argv.end(),[&](Functor* c){ c->parent() = this; });
		delete f;
	}

	inline Functor::Functor(const Functor& f) : _id(f.id()), _par(nullptr)
	{
		argv.reserve(f.argv.size());
		std::for_each(f.argv.begin(),f.argv.end(),[&](Functor* c){
			argv.push_back(c->copy());
			argv.back()->parent() = this;
		});
	}

	
	inline Functor::Functor(Functor&& f) : _id(f.id()), _par(nullptr), argv(std::move(f.argv))
	{
		std::for_each(argv.begin(),argv.end(),[&](Functor* c){ c->parent() = this; });
	}

	inline Functor::~Functor()
	{
		// Recursively calls destructors
		std::for_each(argv.begin(),argv.end(),[&](Functor* c){ delete c; });


		// Non-recursive definition
		//auto i = post_begin();
		//auto next_i = std::next(i);
		//while (&*i != this) {
		//	delete &*i;
		//	i = next_i;
		//	++next_i;
		//}
	}

	inline void Functor::clear()
	{
		std::for_each(arg_begin(),arg_end(),[](Functor* c){ delete c; });
		argv.clear();
		_par = nullptr;
	}
	
	inline Functor* Functor::top()
	{
		Functor* p = this;
		while (p->parent()) p = p->parent(); // move up
		return p;
	}

	inline int Functor::size() const 
	{ 
		return std::distance(begin(),end()); 
	}

	inline long long Functor::to_unsigned() const
	{
		const auto i = this->to_int();
		if (i < 0) throw conversion_error();
		// positive int cannot overflow unsigned
		//if (i > numeric_limits<unsigned>::max()) 
		//	throw std::range_error("prolog int does not fit in unsigned type");
		return i;
	}

	inline void Functor::remove_parent_pointer(Functor* n)
	{
		assert(n != nullptr);
		Functor* npar = n->parent();
		if (npar) {
			// bool found = false;
			for (auto i = npar->arg_begin(); /*i != npar->arg_end()*/; ++i) {
				if (*i == n) {
					npar->argv.erase(i);
					//found = true;
					break;
				}
			}
			//if (!found) {
			//	cerr << "Error: no parent pointer for " << *n << "\n";
			//	cerr << "Parent is: " << *npar << "\n";
			//	assert(false);
			//}
		}
	}

	inline void Functor::steal(Functor* f)
	{
		remove_parent_pointer(f);
		f->parent() = this;
		argv.push_back(f);
	}

	
	inline void Functor::steal(Functor* f, arg_iterator i)
	{
		remove_parent_pointer(f);
		f->parent() = this;
		argv.insert(i,f);
	}

	inline Functor* Functor::detach()
	{
		remove_parent_pointer(this);
		this->parent() = nullptr;
		return this;
	}

	inline bool Functor::is_ground() const
	{
		return std::none_of(begin(),end(),[](const Functor& n){ return n.is_variable(); });
	}

	inline Functor::arity_type Functor::literals() const { return (head() != nullptr) + body_size(); }

	inline void Functor::head_clear()
	{
		if (is_fact()) {
			// Make empty goal
			id(if_id);
			arg_clear();
		} else {
			assert(is_rule() || is_goal());
			arg_pop_first();
		}
	}

	inline const Functor* Functor::head() const
	{
		if (is_fact()) return this;
		else if (is_rule()) return arg_first();
		else return nullptr;
	}

	inline const Functor* Functor::body() const
	{
		if (id() == if_id && (arity() == 1 || arity() == 2)) return arg_last();
		else return nullptr;
	}

	inline void Functor::body_clear()
	{
		if (is_goal()) {
			arg_pop_last();
		} else if (is_rule()) {
			arg_pop_last();
			// Move up head: :- h => h
			id(arg_first()->id());
			vec_type hargs = move(arg_first()->argv);
			delete arg_first();
			argv = move(hargs);
			for_each(argv.begin(),argv.end(),[&](Functor* c){c->parent() = this;});
		}
		// else: is_fact(), do nothing
	}

	inline const Functor* Functor::body_front() const
	{
		assert(!is_empty_goal() && is_goal() || is_rule());
		return (arg_last()->sign(sequence_id,2)) ? arg_last()->arg_first() : arg_last();
	}
	
	inline Functor* Functor::body_front()
	{
		assert(!is_empty_goal() && is_goal() || is_rule());
		return (arg_last()->sign(sequence_id,2)) ? arg_last()->arg_first() : arg_last();
	}

	
	inline const Functor* Functor::body_back() const
	{
		assert(!is_empty_goal() && is_goal() || is_rule());
		const Functor* ptr = arg_last();
		while (ptr->sign(sequence_id,2)) ptr = ptr->arg_last();
		return ptr;
	}

	
	inline Functor* Functor::body_back()
	{
		assert(!is_empty_goal() && is_goal() || is_rule());
		Functor* ptr = arg_last();
		while (ptr->sign(sequence_id,2)) ptr = ptr->arg_last();
		return ptr;
	}

	inline Functor* Functor::body_release_front() { return body_release(seq_iterator(body_front())); }
	inline Functor* Functor::body_release_back() { return body_release(seq_iterator(body_back())); }

	// Iterators
	inline Functor::seq_const_iterator Functor::body_begin() const
	{
		if (id() == if_id && (arity() == 1 || arity() == 2)) {
			// Non-empty goal (ar=1) or rule (ar=2)
			return seq_const_iterator(arg_last());
		} else return seq_const_iterator(nullptr); // fact or empty goal
	}

	
	inline Functor::seq_iterator Functor::body_begin()
	{
		if (id() == if_id && (arity() == 1 || arity() == 2)) {
			// Non-empty goal (ar=1) or rule (ar=2)
			return seq_iterator(arg_last());
		} else return seq_iterator(nullptr); // fact or empty goal
	}
	
	inline Functor::seq_const_iterator Functor::body_end() const { return seq_const_iterator(nullptr); }
	inline Functor::seq_iterator Functor::body_end() { return seq_iterator(nullptr); }

	inline Functor::depth_iterator Functor::begin() { return depth_begin(); }
	inline Functor::depth_iterator Functor::end() { return depth_end(); }
	inline Functor::depth_const_iterator Functor::begin() const { return depth_begin(); }
	inline Functor::depth_const_iterator Functor::end() const { return depth_end(); }
	inline Functor::depth_const_iterator Functor::cbegin() const { return depth_cbegin(); }
	inline Functor::depth_const_iterator Functor::cend() const { return depth_cend(); }

	inline Functor::depth_const_iterator Functor::depth_begin() const { return depth_const_iterator(this); }
	inline Functor::depth_const_iterator Functor::depth_end() const { return depth_const_iterator(nullptr,this); }
	inline Functor::depth_iterator Functor::depth_begin() { return depth_iterator(this); }
	inline Functor::depth_iterator Functor::depth_end() { return depth_iterator(nullptr,this); }
	inline Functor::depth_const_iterator Functor::depth_cbegin() const { return depth_const_iterator(this); }
	inline Functor::depth_const_iterator Functor::depth_cend() const { return depth_const_iterator(nullptr,this); }

	inline Functor::post_const_iterator Functor::post_begin() const { return post_const_iterator(this); }
	inline Functor::post_const_iterator Functor::post_end() const { return post_const_iterator(nullptr,this); }
	inline Functor::post_iterator Functor::post_begin() { return post_iterator(this); }
	inline Functor::post_iterator Functor::post_end() { return post_iterator(nullptr,this); }
	inline Functor::post_const_iterator Functor::post_cbegin() const { return post_const_iterator(this); }
	inline Functor::post_const_iterator Functor::post_cend() const { return post_const_iterator(nullptr,this); }

	// Sequence iterators
	inline Functor::seq_const_iterator Functor::seq_cbegin() const { return seq_const_iterator(this); }
	inline Functor::seq_const_iterator Functor::seq_cend() const { return seq_const_iterator(nullptr); }
	inline Functor::seq_const_iterator Functor::seq_begin() const { return seq_const_iterator(this); }
	inline Functor::seq_const_iterator Functor::seq_end() const { return seq_const_iterator(nullptr); }
	inline Functor::seq_iterator Functor::seq_begin() { return seq_iterator(this); }
	inline Functor::seq_iterator Functor::seq_end() { return seq_iterator(nullptr); }

	// List iterators
	inline Functor::list_const_iterator Functor::list_cbegin() const { return list_const_iterator(this); }
	inline Functor::list_const_iterator Functor::list_cend() const { return list_const_iterator(nullptr); }
	inline Functor::list_const_iterator Functor::list_begin() const { return list_const_iterator(this); }
	inline Functor::list_const_iterator Functor::list_end() const { return list_const_iterator(nullptr); }
	inline Functor::list_iterator Functor::list_begin() { return list_iterator(this); }
	inline Functor::list_iterator Functor::list_end() { return list_iterator(nullptr); }

	// p/1 iterators, arg_first down
	inline Functor::p1f_const_iterator Functor::p1f_cbegin() const { return p1f_const_iterator(this); }
	inline Functor::p1f_const_iterator Functor::p1f_cend() const { return p1f_const_iterator(nullptr,this); }
	inline Functor::p1f_const_iterator Functor::p1f_begin() const { return p1f_const_iterator(this); }
	inline Functor::p1f_const_iterator Functor::p1f_end() const { return p1f_const_iterator(nullptr,this); }
	inline Functor::p1f_iterator Functor::p1f_begin() { return p1f_iterator(this); }
	inline Functor::p1f_iterator Functor::p1f_end() { return p1f_iterator(nullptr,this); }

	// breadth_iterator members
	//inline Functor::breadth_const_iterator::breadth_const_iterator() : nptr(nullptr) { }
	//inline Functor::breadth_const_iterator::breadth_const_iterator(nullptr_t) : nptr(nullptr) { }
	//inline Functor::breadth_const_iterator::breadth_const_iterator(const Functor* ftor) : nptr(ftor) 
	//{
	//	if (!nptr) return;
	//	for_each(nptr->arg_begin(),nptr->arg_end(),[&](const Functor* f){ q.push(f); });
	//}

	//inline Functor::breadth_const_iterator& Functor::breadth_const_iterator::operator++() 
	//{
	//	if (q.empty()) {
	//		// point at end()
	//		nptr = nullptr;
	//		return *this;
	//	}
	//	nptr = q.front();
	//	q.pop();
	//	for_each(nptr->arg_begin(),nptr->arg_end(),[&](const Functor* f){ q.push(f); });
	//	return *this;
	//}

	//inline Functor::breadth_const_iterator Functor::breadth_const_iterator::operator++(int) 
	//{
	//	breadth_const_iterator tmp = *this;
	//	++(*this);
	//	return tmp;
	//}


} // namespace lp



#endif

